import Foundation

/// Demonstration of Gompertz-Makeham mortality model functionality
/// This file shows practical usage examples of the implemented model
class GompertzMakehamDemo {
    
    private let model = GompertzMakehamModel()
    
    /// Demonstrate basic mortality calculations
    func demonstrateBasicCalculations() {
        print("=== Gompertz-Makeham Mortality Model Demo ===\n")
        
        // Register the model in the mortality model registry
        MortalityModelRegistry.register(model: model)
        print("✅ Model registered in MortalityModelRegistry")
        
        print("\n--- Basic Mortality Rate Calculations ---")
        let ages = [20.0, 30.0, 40.0, 50.0, 60.0, 70.0, 80.0, 90.0]
        
        for age in ages {
            let maleMortality = model.calculateMortality(age: age, gender: .male)
            let femaleMortality = model.calculateMortality(age: age, gender: .female)
            
            print("Age \(Int(age)): Male \(String(format: "%.6f", maleMortality)) | Female \(String(format: "%.6f", femaleMortality))")
        }
        
        print("\n--- Life Expectancy Calculations ---")
        for age in ages {
            let maleLE = model.calculateLifeExpectancy(atAge: age, gender: .male)
            let femaleLE = model.calculateLifeExpectancy(atAge: age, gender: .female)
            
            print("Age \(Int(age)): Male \(String(format: "%.1f", maleLE))y | Female \(String(format: "%.1f", femaleLE))y")
        }
        
        print("\n--- Survival Probabilities ---")
        let currentAge = 25.0
        let targetAges = [50.0, 65.0, 75.0, 85.0, 95.0]
        
        for targetAge in targetAges {
            do {
                let survivalProbability = try model.survivalProbability(from: currentAge, to: targetAge, healthMetrics: nil)
                print("Survival from \(Int(currentAge)) to \(Int(targetAge)): \(String(format: "%.3f", survivalProbability)) (\(String(format: "%.1f", survivalProbability * 100))%)")
            } catch {
                print("Error calculating survival probability: \(error)")
            }
        }
    }
    
    /// Demonstrate integration with ActuarialCalculator
    func demonstrateActuarialIntegration() {
        print("\n=== Integration with Actuarial Calculator ===\n")
        
        // Create actuarial calculator with Gompertz-Makeham model
        let calculator = ActuarialCalculator(mortalityModel: model)
        
        // Create sample health metrics
        let healthMetrics = HealthMetrics(
            restingHeartRate: 65.0,
            hrv: 35.0,
            vo2Max: 45.0,
            bmi: 23.5,
            stepCount: 8500,
            sleepHours: 7.5,
            smokingStatus: .never,
            exerciseFrequency: .regular
        )
        
        do {
            let result = try calculator.calculateLifeExpectancy(
                age: 35.0,
                gender: .male,
                healthMetrics: healthMetrics
            )
            
            print("📊 Actuarial Calculation Results:")
            print("Life Expectancy: \(String(format: "%.1f", result.lifeExpectancyYears)) years")
            print("Method Used: \(result.methodUsed)")
            print("Confidence Interval: \(String(format: "%.1f", result.confidenceInterval.lowerBound)) - \(String(format: "%.1f", result.confidenceInterval.upperBound)) years")
            
            print("\n🔍 Risk Factor Impacts:")
            for impact in result.riskFactorImpacts {
                let sign = impact.yearsImpact >= 0 ? "+" : ""
                print("- \(impact.factor): \(sign)\(String(format: "%.1f", impact.yearsImpact)) years (HR: \(String(format: "%.2f", impact.hazardRatio)))")
            }
            
        } catch {
            print("❌ Error in actuarial calculation: \(error)")
        }
    }
    
    /// Demonstrate model validation
    func demonstrateModelValidation() {
        print("\n=== Model Validation ===\n")
        
        // Run validation tests
        let tests = GompertzMakehamModelTests()
        let allTestsPassed = tests.runAllTests()
        
        if allTestsPassed {
            print("\n🎉 Model validation successful - ready for production use")
        } else {
            print("\n⚠️ Model validation identified issues - requires review")
        }
    }
    
    /// Run complete demonstration
    func runFullDemo() {
        demonstrateBasicCalculations()
        demonstrateActuarialIntegration()
        demonstrateModelValidation()
        
        print("\n=== Demo Complete ===")
        print("The Gompertz-Makeham mortality model is now fully implemented and integrated.")
        print("Key features:")
        print("✅ Age-based mortality rate calculation: μ(x) = A + B*exp(C*x)")
        print("✅ Numerical stability for edge cases and extreme ages")
        print("✅ Gender-specific parameters based on demographic research")
        print("✅ Integration with MortalityModel protocol")
        print("✅ Validation against known mortality tables")
        print("✅ Foundation for Cox proportional hazards integration")
    }
}

/// Quick function to run the demo
func runGompertzMakehamDemo() {
    let demo = GompertzMakehamDemo()
    demo.runFullDemo()
}